/****************************************************************************************/
/*                                                                                      */
/* File:        tsfw_sci.c               Version: 003                                   */
/*                                                                                      */
/* Author:      Gerhard Ort                                                             */
/*                                                                                      */
/* Language:    C                                                                       */
/*                                                                                      */
/* Description: Code of the TS-FW SCPI Command Interpreter                              */
/*              based on BIST_SCI.C from BIST_Software for mcb2, STATE004               */
/*                                                                                      */
/****************************************************************************************/
/*                                                                                      */
/* 18.09.98 V001 Gerhard Ort File created                                               */
/* 14.02.2004 V003 Vpr new foranted                                                     */
/*                                                                                      */
/****************************************************************************************/
/****************************************************************************************/
/* Imports                                                                              */
/****************************************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "SystemGlobals.h"
#include "SystemCommandInterpreter_core.h"

extern char tTSFW_SCI_Error_String[];                   // defined in tsfw_sys.c
// functions with semaphores, used in fSetError and fResetError
unsigned char fTSFW_ErrorCanBeSet(unsigned char ucSemaphoreNumber);
void fTSFW_ResetErrorSemaphore(unsigned char ucSemaphoreNumber);
/****************************************************************************************/
/* Exports                                                                              */
/****************************************************************************************/
#include "SystemSerialInterface.h"
unsigned char fSetError (unsigned short ushErrNo, char *pcErrStr);
/*--------------------------------------------------------------------------------------*/
/* Prototypes of Functions for Command Interpreter Functions: Commands and Queries      */
/*--------------------------------------------------------------------------------------*/
static unsigned short fErrorQuery (char *dummy);
static unsigned short fIdentificationQuery (char *dummy);
static unsigned short fActivateDebugMode (char *dummy);
static unsigned short fDeactivateDebugMode (char *dummy);
static unsigned short fHelp (char *pcParm);

COM_DEF asSCPI_ComDef [] =
{
    /*----------------------------------------------------------------------------------*/
    /*  Basic functions of SCPI                                                         */
    /*----------------------------------------------------------------------------------*/
    {":SYST:ERR?",       fErrorQuery},
    {"*IDN?",            fIdentificationQuery},
    {"*DEBUG",           fActivateDebugMode},
    {"*NODEBUG",         fDeactivateDebugMode},
    {":HELP?",           fHelp},
    {NULL,               NULL}		// array must end with NULL
};
// global Flag, if Testsoftware has been started, modules and the active communication
// channel are init
unsigned char ucTSFW_Init_Ready=0;
/****************************************************************************************/
/* Locals                                                                               */
/****************************************************************************************/
#define     DEBUG_ERR           0x01
#define     NO_DEBUG_ERR        0x00
/*--------------------------------------------------------------------------------------*/
/* ASCII Codes                                                                          */
/*--------------------------------------------------------------------------------------*/
#define     SPACE               0x20
#define     ASCII_Q             0x3f
/*--------------------------------------------------------------------------------------*/
/* modulglobal type of next communication to use                                        */
/*--------------------------------------------------------------------------------------*/
static unsigned char ucGlobalComNew, ucGlobalCom2Change;
/*--------------------------------------------------------------------------------------*/
/* modulglobal error status                                                             */
/*--------------------------------------------------------------------------------------*/
static struct {
    unsigned char       ucFlag;                 // Error Flag
    unsigned short      ushNo;                  // Error Number
    char                *pcStr;                 // Error String
    unsigned char       ucDebug;                // Debug Flag
} sErr = { 0, 0, NULL, NO_DEBUG_ERR };
/*--------------------------------------------------------------------------------------*/
/* Prototypes other Functions                                                           */
/*--------------------------------------------------------------------------------------*/
static char *fGetParmString (char *pcCom);
static unsigned char fCompareCom (char *pcS1, char *pcS2);
static unsigned short fRunTsfwFunc (char *pcCom, char *pcParm);
static void fResetError (void);
/****************************************************************************************/
/* Functions                                                                            */
/****************************************************************************************/
/***************************************************************************************/
/*                                                                                      */
/* Function   : fTSFW_InitPCB()                                                         */
/*                                                                                      */
/* Description: Calls all init fuctions of the modules, calls nothing on further calls  */
/*                                                                                      */
/* Return     : none                                                                    */
/*                                                                                      */
/****************************************************************************************/
void fTSFW_InitPCB (void){
static unsigned char ucInit=0;
void (**ppfInitFuncList)(void)=apfInitFuncList;
if(ucInit) return;
ucInit = 1;
// Init local pointer variable to string variable, which is defined in tsfw_sys.c
sErr.pcStr = tTSFW_SCI_Error_String;
// init start mode of debug error outputs
sErr.ucDebug = ucTSFW_DEBUG_START_MODE;
// execute all functions in InitFuncList
while(*ppfInitFuncList != NULL) {
  (**ppfInitFuncList)();
  ppfInitFuncList++;
  }
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fTSFW_CommandInterpreter()                                              */
/*                                                                                      */
/* Description: function, receives one command from communication channel, calls the    */
/*              TS-FW function and outputs answers                                      */
/*                                                                                      */
/* Return     : 0: O.K. (command executed or there was no command to execute)           */
/*              !=0: Error from communication channel                                   */
/*                                                                                      */
/****************************************************************************************/
unsigned short fTSFW_CommandInterpreter (void){
extern char tTSFW_SCI_CommandInterpreter_String[];      // defined in tsfw_sys.c
char                *pc=tTSFW_SCI_CommandInterpreter_String+1, *pcCom, *pcParm;
unsigned short      ush, ushAns;
unsigned char       ucQuery, i;
if(ucTSFW_Init_Ready==0){
  // Init TSFW-modules
  fTSFW_InitPCB();
  // Init selected communication module
  strcpy(pc, pcTSFW_DEFAULT_COMMUNICATION_COMMAND);
  pcCom = pc;
  pcParm = fGetParmString (pcCom);    // get Parameter String and cut it from pcCom
  fToUpperCase (pcCom);               // convert command to upper case
  ush= fRunTsfwFunc (pcCom, pcParm);
  if(ush) {
    fSetError (ush, "Communication Init Error");    // if error output it on PCB
    return 1;
    }
  if(!ucGlobalCom2Change) {
    fSetError (ush, "Communication Init Error Com2Change");    // if error output it on PCB
    return 4;
    }
  ush = (*(asTciChannelList[ucGlobalComNew].pfChange))();
  if(ush) {
    fSetError (ush, "Communication error"); // if error, output it on PCB
    return 2;
    }
  ucTSFW_GlobalCom = ucGlobalComNew;
  ucTSFW_Init_Ready = TRUE;
  if(ucDEBUG) {
    for(i=0; apfBufferClearanceFunctionList[i] != NULL; i++)
      (*(apfBufferClearanceFunctionList[i]))();
    (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) ("Eingeschaltet oder Reset");
    }
  return 0;   // it's enough for the first start
  }
    /*------------------------------------------------------------------------------*/
    /* read new command string, first                                               */
    /*------------------------------------------------------------------------------*/
 ush = (*(asTciChannelList[ucTSFW_GlobalCom].pfReceiveString)) (pc);
 if ( ush ) {
  if(ush == COM_NOT_FINISHED) return 0;   // no error, there was no command
  fSetError (ush, "Communication Error");     // if error output it on PCB
  return 3;
  }
i = strlen(pc);

  if (i < 2) { fSetError (SYNTAX_ERR, "Unknown Command/Query, too short"); }// a valid command has at least 3 char
  else {
    /*--------------------------------------------------------------------------*/
    /* analize command string                                                   */
    /*--------------------------------------------------------------------------*/
    if ( *pc == ASCII_Q ) {
      // first character is '?'
      ucQuery = TRUE;
      pcCom = pc+1;
      }
    else
      if( pc[i-1] == ASCII_Q ) {
        // last character is '?'
        ucQuery = TRUE;
        pc[i-1] = '\0';
        pcCom = pc;
        }
      else {
        ucQuery = FALSE;
        pcCom = pc;     // no '?' found, it's a command not a query
        }
      pcParm = fGetParmString (pcCom);    // get Parameter String and cut it from pcCom
     fToUpperCase (pcCom);               // convert command to upper case
      if(ucQuery) {
        // Command-String should end with '?' -> all characters must be copied
        // for one position to get place for the '?'
        pcCom--;
        for(pc=pcCom; ( *pc=*(pc+1) ); pc++);
        *pc = '?';       // last character after command should be '?'
        // reset pc for next command/query-receive
        pc = tTSFW_SCI_CommandInterpreter_String+1;
        }
      /*--------------------------------------------------------------------------*/
      /* call function and analize answer                                         */
      /*--------------------------------------------------------------------------*/
      ucGlobalCom2Change = 0;
      if (fCompareCom (pcCom, ":SYST:ERR?")) ushAns = fErrorQuery(NULL);  // Error Query? must allways work !
      else if (!sErr.ucFlag){                      // if no error
        ushAns = fRunTsfwFunc (pcCom, pcParm);  // run TS-FW function if existent
        if( ushAns == NO_ERR ) {
          if(ucQuery == FALSE) pcReturn="H"; // Command answer, when all O.K.
          }
        else fSetError(ushAns, pcReturn);
        }
      /*--------------------------------------------------------------------------*/
      /* output answer                                                            */
      /*--------------------------------------------------------------------------*/
      if (!sErr.ucFlag) {
        for(i=0; apfBufferClearanceFunctionList[i] != NULL; i++)
          (*(apfBufferClearanceFunctionList[i]))();
         ush = (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (pcReturn);   
         if ( ush ) {
          fSetError (ush, "Communication error"); // if error, output it on PCB
          return 4;
          }
        if(ucGlobalCom2Change) {
          // change the communication channel
          // the init function was called successfully before
          ush = (*(asTciChannelList[ucGlobalComNew].pfChange))(); 
          if (ush) {
//          if ( ush = (*(asTciChannelList[ucGlobalComNew].pfChange))() ) {
            fSetError (ush, "Communication error"); // if error, output it on PCB
            return 2;
            }
          ucTSFW_GlobalCom = ucGlobalComNew;  // from now on all goes to new com-port
          }
       }
    }
  /*------------------------------------------------------------------------------*/
  /* output and reset error in debug mode                                         */
  /*------------------------------------------------------------------------------*/
 if ( sErr.ucDebug == DEBUG_ERR && sErr.ucFlag ) {
    fErrorQuery(NULL);
    for(i=0; apfBufferClearanceFunctionList[i] != NULL; i++)
      (*(apfBufferClearanceFunctionList[i]))();
    (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (pcReturn);
    }
  return 0;
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fGetParmString()                                                        */
/*                                                                                      */
/* Description: Searches in pcCom for the first space to find the beginning of the      */
/*              parameter string. If found, the space is replaced by '\0' (pcCom is     */
/*              cut).                                                                   */
/*                                                                                      */
/* Return     : parameter string, if no space was found: empty string                   */
/*                                                                                      */
/****************************************************************************************/
static char *fGetParmString
(
  char *pcCom                  // I/E: Command String
){
while(*pcCom != SPACE)
  if(*pcCom == '\0') return pcCom;   // return address of end sign of command string (empty string)
  else pcCom++;
*pcCom = '\0';          // end command/query name string
return pcCom+1;         // the char after is start of parameter string
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fCompareCom()                                                           */
/*                                                                                      */
/* Description: compares two command strings, the second is converted to uppercase      */
/*                                                                                      */
/* Return     : 1 if equal, 0 if not                                                    */
/*                                                                                      */
/****************************************************************************************/
static unsigned char fCompareCom
(
  char *pcS1,               // IMP: string 1
  char *pcS2                // IMP: string 2
){
char c;
if(pcS1==NULL){ if(pcS2==NULL)  return 1;
                else            return 0;
			   }
if(pcS2==NULL) return 0;
while (*pcS1) {
 c = toupper(*pcS2);
 pcS2++;
 if(*pcS1++ != c) return (0);           // until end of string t1
 } 
if (*pcS2) return (0);
else       return (1);                  // no difference found
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fRunTsfwFunc()                                                          */
/*                                                                                      */
/* Description: searches in the asComDef List for the command pcCom; and if found       */
/*              runs the matching TS-FW function                                        */
/*                                                                                      */
/* Return     : no_error or error number and query answer string of the TS-FW function  */
/*              if the TS-FW function was found, otherwise 'Unknown query/error' error  */
/*              string                                                                  */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fRunTsfwFunc
(
  char *pcCom,             // IMP: Command String
  char *pcParm             // IMP: Parameter String
){
COM_DEF_LIST    *psComDefList = asComDefList;
COM_DEF         *psComDef;
unsigned char   ucFound = FALSE;
// search first for SCPI-Commands (have most priority)
psComDef=asSCPI_ComDef;
while(psComDef->pcCommand != NULL && !ucFound) {
  if(fCompareCom(pcCom, psComDef->pcCommand)) ucFound = TRUE;
  else  psComDef++;
  }
// search in Communication module list
if(!ucFound){
  ucGlobalComNew = 0;
  while(!ucFound && asTciChannelList[ucGlobalComNew].pcCommand != NULL){
    if(fCompareCom(pcCom, asTciChannelList[ucGlobalComNew].pcCommand)){
      psComDef = (COM_DEF *)&asTciChannelList[ucGlobalComNew];
      ucFound = TRUE;
      }
    else ucGlobalComNew++;
    }
  if(ucFound) ucGlobalCom2Change = 1;
  }
// search in Command-Lists of modules, if command was not yet found
while(!ucFound && psComDefList->pasComDef != NULL) {
  psComDef=psComDefList->pasComDef;
  // search in specific list
  while(psComDef->pcCommand != NULL && !ucFound) {
    if(fCompareCom(pcCom, psComDef->pcCommand)) ucFound = TRUE;
    else psComDef++;
    }
  psComDefList++;
  }
if(ucFound){
  return ((* (psComDef->callfunc)) (pcParm)); // call TS-FW function and return result
  }
  else if(pcCom[strlen(pcCom)-1] == '?') {
         RETURN( SYNTAX_ERR, "Unknown Query" );      // query not found
         }
       else {
         RETURN( SYNTAX_ERR, "Unknown Command" );    // command not found
    }
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fSetError()                                                             */
/*                                                                                      */
/* Description: Function to set an error                                                */
/*                                                                                      */
/* Return     : 1 if an error was already set, otherwise 0                              */
/*                                                                                      */
/****************************************************************************************/
unsigned char fSetError
(
unsigned short  ushErrNo,    // IMP: Error Number
char            *pcErrStr    // IMP: Error String
)
{
typedef void (**ppfSOFL_T)(unsigned short);
ppfSOFL_T ppfSOFL = apfStatusOutputFunctionList;
unsigned char ucCount;
// set only once an error:
if(!fTSFW_ErrorCanBeSet(0)) return 1;
// disable ASSERT-errors, when a error is set
fTSFW_ErrorCanBeSet(1);
if (sErr.ucFlag) return 1;
else {
  sErr.ushNo  = ushErrNo;
  // copy max. ucCOM_STRLEN_MAX characters of the Error String to sErr.tStr
  for(ucCount=0; pcErrStr[ucCount]!='\0' && ucCount<ucCOM_STRLEN_MAX; ucCount++)
    sErr.pcStr[ucCount] = pcErrStr[ucCount];
  // end sErr.tStr with '\0'
  sErr.pcStr[ucCount] = '\0';
  // Output error on FBG
  while(*ppfSOFL!=NULL){
    (**ppfSOFL)(ushErrNo);
    ppfSOFL++;
    }
  sErr.ucFlag = 1;
  return 0;
  }
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fResetError()                                                           */
/*                                                                                      */
/* Description: Function to reset errorflag and to call fbg-specific functions to reset */
/*              their Status output                                                     */
/*                                                                                      */
/* Return     : none                                                                    */
/*                                                                                      */
/****************************************************************************************/
static void fResetError(void)
{
typedef void (**ppfSOFL_T)(unsigned short);
ppfSOFL_T ppfSOFL = apfStatusOutputFunctionList;
if(!sErr.ucFlag) return;     // do nothing, if fSetError was not called before
sErr.ushNo  = NO_ERR;
// Output 'No error' on FBG
while(*ppfSOFL!=NULL) {
  (**ppfSOFL)(NO_ERR);
  ppfSOFL++;
  }
sErr.ucFlag = 0;
// reset Error semaphores for next error to be set
fTSFW_ResetErrorSemaphore(0);
fTSFW_ResetErrorSemaphore(1);   // semaphore of ASSERT functions
}
/*--------------------------------------------------------------------------------------*/
/*                         Command- and Query- Functions                                */
/*--------------------------------------------------------------------------------------*/
/****************************************************************************************/
/*                                                                                      */
/* Function   : fErrorQuery()       > Query <                                           */
/*                                                                                      */
/* Description: This function handles the ":SYST:ERR?" command                          */
/*                                                                                      */
/* Return     : answer string to the error query with the error message                 */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fErrorQuery (char *dummy)
{
dummy = NULL;                                       // avoid unused warning
if (!sErr.ucFlag) strcpy (tMSG_Buffer, "No Error"); // Check Error Flag
else {
  sprintf (tMSG_Buffer, "Error %u: %s", sErr.ushNo, sErr.pcStr);  // build Error message
  fResetError ();             // clear Error Flag and reset output on PCB
  }
RETURN( NO_ERR, tMSG_Buffer );
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fIdentificationQuery()       > Query <                                  */
/*                                                                                      */
/* Description: TS-FW function to answer the standard identification query              */
/*                                                                                      */
/* Return     : answer string to ID query                                               */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fIdentificationQuery (char *dummy)
{
dummy = NULL;                   // avoid unused warning
RETURN( NO_ERR, (char *)IDN_STRING );
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fActivateDebugMode()    > Command <                                     */
/*                                                                                      */
/* Description: sets the global flag to activate the debug mode.                        */
/*              Debug mode means that every error is forwarded to serial port/can       */
/*              the global error status is cleared automatically.                       */
/*                                                                                      */
/* Return     : empty string                                                            */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fActivateDebugMode (char *dummy)
{
dummy = NULL;                   // avoid unused warning
sErr.ucDebug = DEBUG_ERR;
RETURN( NO_ERR, NULL );
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fDeactivateDebugMode()    > Command <                                   */
/*                                                                                      */
/* Description: resets the global flag to deactivate the debug mode                     */
/*              Debug mode means that every error is forwarded to serial port/can and   */
/*              the global error status is cleared automatically.                       */
/*                                                                                      */
/* Return     : empty string                                                            */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fDeactivateDebugMode (char *dummy)
{
dummy = NULL;                   // avoid unused warning
sErr.ucDebug = NO_DEBUG_ERR;
RETURN( NO_ERR, NULL );
}
/****************************************************************************************/
/*                                                                                      */
/* Function   : fHelp         > Query <                                                 */
/*                                                                                      */
/* Description: TS-FW function to give an overview of all possible commands and queries */
/*              they are listed with modul-name as header and separated by commands     */
/*              and queries                                                             */
/*                                                                                      */
/* Return     : none                                                                    */
/*                                                                                      */
/****************************************************************************************/
static unsigned short fHelp
(
  char *pcParm          // IMP: parameter (string)
                        //      when any parameter - no white spaces - is given a long list
                        //      with commands and queries is output, else it's a shorter list
){
COM_DEF_LIST    *psComDefList;
COM_DEF         *psComDef, *psComDefStart;
char            *pcModuleName = "SCPI";
unsigned char   ucStart = FALSE;
unsigned char   ucLongHelp;
static char     tHeader[50];

while(*pcParm && *pcParm == SPACE) pcParm++;
if(*pcParm) ucLongHelp = 1;
else ucLongHelp = 0;
// search first for SCPI-Commands (have most priority)
psComDef = psComDefStart = asSCPI_ComDef;
do {
  if(ucLongHelp) {
    // search for commands (last character is not '?')
    // and list it with a headline
    while(psComDef->pcCommand != NULL
          && (psComDef->pcCommand)[strlen(psComDef->pcCommand)-1] == '?'){
      psComDef++;
      }
    if(psComDef->pcCommand != NULL){
      sprintf(tHeader, "\rCommands in Module %s", pcModuleName);
      (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (tHeader);
      for(; psComDef->pcCommand != NULL; psComDef++) {
        if((psComDef->pcCommand)[strlen(psComDef->pcCommand)-1] != '?') {
          (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (psComDef->pcCommand);
          }
        }
      }
   // search for queries and list it with a headline
    psComDef = psComDefStart;
    while(psComDef->pcCommand != NULL
          && (psComDef->pcCommand)[strlen(psComDef->pcCommand)-1] != '?'){
      psComDef++;
      }
    if(psComDef->pcCommand != NULL){
      sprintf(tHeader, "\rQueries in Module %s", pcModuleName);
      (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (tHeader);
      for(; psComDef->pcCommand != NULL; psComDef++){
        if((psComDef->pcCommand)[strlen(psComDef->pcCommand)-1] == '?') {
          (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (psComDef->pcCommand);
          }
        }
      }
    }
  else {
    // list all commands and queries in one list - more compact
    psComDef = psComDefStart;
    if(psComDef->pcCommand != NULL){
      sprintf(tHeader, "\rCommands/Queries in Module %s", pcModuleName);
      (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (tHeader);
      for(; psComDef->pcCommand != NULL; psComDef++){
        (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (psComDef->pcCommand);
        }
      }
    }
  if(!ucStart){
    // output communication change commands
    sprintf(tHeader, "\rCommunication Change Commands");
    (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (tHeader);
    for(; asTciChannelList[ucStart].pcCommand != NULL; ucStart++)
      (*(asTciChannelList[ucTSFW_GlobalCom].pfSendString)) (asTciChannelList[ucStart].pcCommand);
    // take first module asComDefList
    psComDefList = asComDefList;
    ucStart = TRUE;
    }
  else psComDefList++;  // take next module in asComDefList
  psComDefStart = psComDefList->pasComDef;
  psComDef = psComDefStart;
  pcModuleName = (char *)psComDefList->pcModulName;
}while(psComDef != NULL);
RETURN( NO_ERR, "" );
}
/*--------------------------------------------------------------------*/
/*  global functions, are exported in Tsfw_sci.h                      */
/*--------------------------------------------------------------------*/
/***********************************************************************/
/*                                                                     */
/* Function   : fToUpperCase()                                         */
/*                                                                     */
/* Description: converts all characters of a string to uppercase       */
/*                                                                     */
/* Return     : none                                                   */
/*                                                                     */
/***********************************************************************/
void fToUpperCase
(
  char *pc                      // I/E: string to convert
){
if(pc==NULL)return;

while (*pc = ( toupper (*pc))) pc++; // toupper() is reentrant        
}
